using System;
using Crestron.SimplSharp;                          	// For Basic SIMPL# Classes
using Crestron.SimplSharpPro;                       	// For Basic SIMPL#Pro classes
using Crestron.SimplSharpPro.CrestronThread;        	// For Threading
using Crestron.SimplSharpPro.Diagnostics;		    	// For System Monitor Access
using Crestron.SimplSharpPro.DeviceSupport;         	// For Generic Device Support
using Inogeni;
using Crestron.SimplSharpPro.UI;
using Crestron.SimplSharp.CrestronIO;

namespace Inogeni_CAM300_Demo
{
    public class ControlSystem : CrestronControlSystem
    {
        /// <summary>
        /// ControlSystem Constructor. Starting point for the SIMPL#Pro program.
        /// Use the constructor to:
        /// * Initialize the maximum number of threads (max = 400)
        /// * Register devices
        /// * Register event handlers
        /// * Add Console Commands
        /// 
        /// Please be aware that the constructor needs to exit quickly; if it doesn't
        /// exit in time, the SIMPL#Pro program will exit.
        /// 
        /// You cannot send / receive data in the constructor
        /// </summary>
        /// 

        Tsw760 TP;
        CAM300 C300;
        CAM300 C300_IP;

        public ControlSystem()
            : base()
        {
            try
            {
                Thread.MaxNumberOfUserThreads = 20;

                //Subscribe to the controller events (System, Program, and Ethernet)
                CrestronEnvironment.SystemEventHandler += new SystemEventHandler(ControlSystem_ControllerSystemEventHandler);
                CrestronEnvironment.ProgramStatusEventHandler += new ProgramStatusEventHandler(ControlSystem_ControllerProgramEventHandler);
                CrestronEnvironment.EthernetEventHandler += new EthernetEventHandler(ControlSystem_ControllerEthernetEventHandler);
            }

            catch (Exception e)
            {
                ErrorLog.Error("Error in the constructor: {0}", e.Message);
            }
            
            ComPort SerialPort = null;

            try
            {
                SerialPort = this.ComPorts[1];
                SerialPort.Register();
                C300 = new CAM300(ref SerialPort);
                //SerialPort.SerialDataReceived += new ComPortDataReceivedEvent(OnSerialDataReceived);
            }
            catch
            {
                ErrorLog.Notice("Broke picking serial port.");
            }
            try
            {
                C300_IP = new CAM300("192.168.0.81"); // initialization should force connect to device.
            }
            catch (Exception e)
            {
                ErrorLog.Notice("Broke initializing or connecting to the C300_IP: {0} | {1}",e.Message,e.InnerException);
            }
            try
            {
                TP = new Tsw760(0x03, this);
                TP.LoadSmartObjects(Directory.GetApplicationDirectory() + Path.DirectorySeparatorChar + "Inogeni CAM300 Demo v1.0.sgd");
                TP.Register();
            }
            catch
            {
                ErrorLog.Notice("Error initializing TP.");
            }
        }

        /// <summary>
        /// InitializeSystem - this method gets called after the constructor 
        /// has finished. 
        /// 
        /// Use InitializeSystem to:
        /// * Start threads
        /// * Configure ports, such as serial and verisports
        /// * Start and initialize socket connections
        /// Send initial device configurations
        /// 
        /// Please be aware that InitializeSystem needs to exit quickly also; 
        /// if it doesn't exit in time, the SIMPL#Pro program will exit.
        /// </summary>
        public override void InitializeSystem()
        {
            try
            {
                TP.SigChange += OnTPSigChange;
                TP.SmartObjects[1].SigChange += OnSmartObjectSigChange;
                C300.PropertyChanged += OnCAM300PropertyChanged;
                C300.DebugInfo += OnCAM300Debug;
                C300.RegularPollInterval = 60000;
                C300.EnableRegularPolling = true;
                TP.BooleanInput[20].BoolValue = false;
                TP.StringInput[20].StringValue = "RS232";
            }
            catch (Exception e)
            {
                ErrorLog.Error("Error in InitializeSystem: {0}", e.Message);
            }
            try
            {

                C300_IP.PropertyChanged += OnCAM300PropertyChanged;
                C300_IP.DebugInfo += OnCAM300Debug;
                C300_IP.RegularPollInterval = 60000;
                C300_IP.EnableRegularPolling = true;
            }
            catch (Exception e)
            {
                ErrorLog.Error("Error in InitializeSystem: {0}", e.Message);
            }
        }

        /// <summary>
        /// Event Handler for Ethernet events: Link Up and Link Down. 
        /// Use these events to close / re-open sockets, etc. 
        /// </summary>
        /// <param name="ethernetEventArgs">This parameter holds the values 
        /// such as whether it's a Link Up or Link Down event. It will also indicate 
        /// wich Ethernet adapter this event belongs to.
        /// </param>
        void ControlSystem_ControllerEthernetEventHandler(EthernetEventArgs ethernetEventArgs)
        {
            switch (ethernetEventArgs.EthernetEventType)
            {//Determine the event type Link Up or Link Down
                case (eEthernetEventType.LinkDown):
                    //Next need to determine which adapter the event is for. 
                    //LAN is the adapter is the port connected to external networks.
                    if (ethernetEventArgs.EthernetAdapter == EthernetAdapterType.EthernetLANAdapter)
                    {
                        //
                    }
                    break;
                case (eEthernetEventType.LinkUp):
                    if (ethernetEventArgs.EthernetAdapter == EthernetAdapterType.EthernetLANAdapter)
                    {

                    }
                    break;
            }
        }

        /// <summary>
        /// Event Handler for Programmatic events: Stop, Pause, Resume.
        /// Use this event to clean up when a program is stopping, pausing, and resuming.
        /// This event only applies to this SIMPL#Pro program, it doesn't receive events
        /// for other programs stopping
        /// </summary>
        /// <param name="programStatusEventType"></param>
        void ControlSystem_ControllerProgramEventHandler(eProgramStatusEventType programStatusEventType)
        {
            switch (programStatusEventType)
            {
                case (eProgramStatusEventType.Paused):
                    //The program has been paused.  Pause all user threads/timers as needed.
                    break;
                case (eProgramStatusEventType.Resumed):
                    //The program has been resumed. Resume all the user threads/timers as needed.
                    break;
                case (eProgramStatusEventType.Stopping):
                    //The program has been stopped.
                    //Close all threads. 
                    //Shutdown all Client/Servers in the system.
                    //General cleanup.
                    //Unsubscribe to all System Monitor events
                    break;
            }

        }

        /// <summary>
        /// Event Handler for system events, Disk Inserted/Ejected, and Reboot
        /// Use this event to clean up when someone types in reboot, or when your SD /USB
        /// removable media is ejected / re-inserted.
        /// </summary>
        /// <param name="systemEventType"></param>
        void ControlSystem_ControllerSystemEventHandler(eSystemEventType systemEventType)
        {
            switch (systemEventType)
            {
                case (eSystemEventType.DiskInserted):
                    //Removable media was detected on the system
                    break;
                case (eSystemEventType.DiskRemoved):
                    //Removable media was detached from the system
                    break;
                case (eSystemEventType.Rebooting):
                    //The system is rebooting. 
                    //Very limited time to preform clean up and save any settings to disk.
                    break;
            }

        }

        void OnTPSigChange(BasicTriList list, SigEventArgs args)
        {
            CAM300 ActiveCAM300;
            switch (TP.BooleanInput[20].BoolValue)
            {
                case (true):
                {
                    ActiveCAM300 = C300_IP;
                    break;
                }
                default:
                {
                    ActiveCAM300 = C300;
                    break;
                }
            }
            switch (args.Sig.Type)
            {
                case(eSigType.Bool):
                {
                    if (args.Sig.BoolValue == true)
                    {
                        switch(args.Sig.Number)
                        {
                            case(11):
                            case(12):
                            case(13):
                            case(14):
                            {
                                ActiveCAM300.SetInput((ushort)(args.Sig.Number-10));
                                break;
                            }
                            case (20):
                            {
                                list.BooleanInput[20].BoolValue = !list.BooleanInput[20].BoolValue;
                                switch (list.BooleanInput[20].BoolValue)
                                {
                                    case(true):
                                    {
                                        list.StringInput[20].StringValue = "IP";
                                        break;
                                    }
                                    default:
                                    {
                                        list.StringInput[20].StringValue = "RS232";
                                        break;
                                    }
                                }
                                break;
                            }
                            case (21): //zoom in
                            {
                                ActiveCAM300.ZoomIn(ActiveCAM300.CurrentInput, 5);
                                break;
                            }
                            case (22)://zoom out
                            {
                                ActiveCAM300.ZoomOut(ActiveCAM300.CurrentInput, 5);
                                break;
                            }
                            case (31):
                            {
                                ActiveCAM300.Blackout();
                                break;
                            }
                            case (32):
                            {
                                ActiveCAM300.UnBlackout();
                                break;
                            }
                            case (61):
                            {
                                ActiveCAM300.SetInversionState[1] = InversionState.NoFlip;
                                break;
                            }
                            case (62):
                            {
                                ActiveCAM300.SetInversionState[1] = InversionState.VerticalFlip;
                                break;
                            }
                            case (63):
                            {
                                ActiveCAM300.SetInversionState[1] = InversionState.HorizontalFlip;
                                break;
                            }
                            case (64):
                            {
                                ActiveCAM300.SetInversionState[2] = InversionState.NoFlip;
                                break;
                            }
                            case (65):
                            {
                                ActiveCAM300.SetInversionState[2] = InversionState.VerticalFlip;
                                break;
                            }
                            case (66):
                            {
                                ActiveCAM300.SetInversionState[2] = InversionState.HorizontalFlip;
                                break;
                            }
                            case (67):
                            {
                                ActiveCAM300.SetInversionState[3] = InversionState.NoFlip;
                                break;
                            }
                            case (68):
                            {
                                ActiveCAM300.SetInversionState[3] = InversionState.VerticalFlip;
                                break;
                            }
                            case (69):
                            {
                                ActiveCAM300.SetInversionState[3] = InversionState.HorizontalFlip;
                                break;
                            }
                            case (70):
                            {
                                ActiveCAM300.SetInversionState[4] = InversionState.NoFlip;
                                break;
                            }
                            case (71):
                            {
                                ActiveCAM300.SetInversionState[4] = InversionState.VerticalFlip;
                                break;
                            }
                            case (72):
                            {
                                ActiveCAM300.SetInversionState[4] = InversionState.HorizontalFlip;
                                break;
                            }
                            default: { break; }
                        }
                    }
                    break;
                }
                default: { break; }
            }
        }
        void OnSmartObjectSigChange(GenericBase device, SmartObjectEventArgs args)
        {
            CAM300 ActiveCAM300;
            switch (TP.BooleanInput[20].BoolValue)
            {
                case (true):
                {
                    ActiveCAM300 = C300_IP;
                    break;
                }
                default:
                {
                    ActiveCAM300 = C300;
                    break;
                }
            }
            switch (args.Sig.Name)
            {
                case ("Up"):
                {
                    //ErrorLog.Notice("Up changed.");
                    switch (args.Sig.BoolValue)
                    {
                        case(true):
                        {
                            ActiveCAM300.TiltUp(ActiveCAM300.CurrentInput);
                            break;
                        }
                        case (false):
                        {
                            ActiveCAM300.StopTilt(ActiveCAM300.CurrentInput);
                            break;
                        }
                    }
                    break;
                }
                case ("Down"):
                {
                    //ErrorLog.Notice("Down changed.");
                    switch (args.Sig.BoolValue)
                    {
                        case (true):
                            {
                                ActiveCAM300.TiltDown(ActiveCAM300.CurrentInput);
                                break;
                            }
                        case (false):
                            {
                                ActiveCAM300.StopTilt(ActiveCAM300.CurrentInput);
                                break;
                            }
                    }
                    break;
                }
                case ("Left"):
                {
                    //ErrorLog.Notice("Left changed.");
                    switch (args.Sig.BoolValue)
                    {
                        case (true):
                            {
                                ActiveCAM300.PanCounterClockwise(ActiveCAM300.CurrentInput);
                                break;
                            }
                        case (false):
                            {
                                ActiveCAM300.StopPan(ActiveCAM300.CurrentInput);
                                break;
                            }
                    }
                    break;
                }
                case ("Right"):
                {
                    //ErrorLog.Notice("Right changed.");
                    switch (args.Sig.BoolValue)
                    {
                        case (true):
                            {
                                ActiveCAM300.PanClockwise(ActiveCAM300.CurrentInput);
                                break;
                            }
                        case (false):
                            {
                                ActiveCAM300.StopPan(ActiveCAM300.CurrentInput);
                                break;
                            }
                    }
                    break;
                }
            }
        }
        void OnCAM300PropertyChanged(object sender, PropertyChangedEventArgs args)
        {
            CAM300 ActiveCAM300;
            switch (TP.BooleanInput[20].BoolValue)
            {
                case (true):
                    {
                        ActiveCAM300 = C300_IP;
                        break;
                    }
                default:
                    {
                        ActiveCAM300 = C300;
                        break;
                    }
            }
            switch (args.PropertyName)
            {
                case ("MACAddress"):
                {
                    TP.StringInput[51].StringValue = (string)args.PropertyValue;
                    break;
                }
                case ("IPAddress"):
                {
                    TP.StringInput[52].StringValue = (string)args.PropertyValue;
                    break;
                }
                case ("StreamerApp"):
                {
                    TP.StringInput[53].StringValue = (string)args.PropertyValue;
                    break;
                }
                case ("CurrentSource"):
                {
                    for (uint i = 1; i < 5; i++)
                    {
                        if ((ushort)args.PropertyValue == (ushort)i)
                        {
                            TP.BooleanInput[i + 10].BoolValue = true;
                        }
                        else
                        {
                            TP.BooleanInput[i + 10].BoolValue = false;
                        }
                    }
                    break;
                }
                case ("Blackout"):
                {
                    if ((bool)args.PropertyValue == true)
                    {
                        TP.BooleanInput[32].BoolValue = false;
                        TP.BooleanInput[31].BoolValue = true;
                    }
                    else
                    {
                        TP.BooleanInput[31].BoolValue = false;
                        TP.BooleanInput[32].BoolValue = true;
                    }
                    break;
                }
                case ("SubnetMask"):
                {
                    TP.StringInput[54].StringValue = (string)args.PropertyValue;
                    break;
                }
                default:
                {
                    break;
                }
            }
            if (args.PropertyName.Contains("InversionStateFB["))
            {
                //ErrorLog.Notice("Inversion state {0} changed to {1}", args.PropertyName.Substring("InversionStateFB[".Length, 1), args.PropertyValue.ToString());
                uint index = Convert.ToUInt32(args.PropertyName.Substring("InversionStateFB[".Length, 1));
                uint[] InversionButtonFBJoins = new uint[3];
                switch (index)
                {
                    case(1):
                    {
                        InversionButtonFBJoins = new uint[3] { 61, 62, 63 };
                        break;
                    }
                    case (2):
                    {
                        InversionButtonFBJoins = new uint[3] { 64, 65, 66 };
                        break;
                    }
                    case (3):
                    {
                        InversionButtonFBJoins = new uint[3] { 67, 68, 69 };
                        break;
                    }
                    case (4):
                    {
                        InversionButtonFBJoins = new uint[3] { 70, 71, 72 };
                        break;
                    }
                }
                switch (ActiveCAM300.InversionStateFB[(int)index])
                {
                    case(InversionState.NoFlip):
                    {
                        TP.BooleanInput[InversionButtonFBJoins[1]].BoolValue = false;
                        TP.BooleanInput[InversionButtonFBJoins[2]].BoolValue = false;
                        TP.BooleanInput[InversionButtonFBJoins[0]].BoolValue = true;
                        break;
                    }
                    case (InversionState.VerticalFlip):
                    {
                        TP.BooleanInput[InversionButtonFBJoins[0]].BoolValue = false;
                        TP.BooleanInput[InversionButtonFBJoins[2]].BoolValue = false;
                        TP.BooleanInput[InversionButtonFBJoins[1]].BoolValue = true;
                        break;
                    }
                    case (InversionState.HorizontalFlip):
                    {
                        TP.BooleanInput[InversionButtonFBJoins[0]].BoolValue = false;
                        TP.BooleanInput[InversionButtonFBJoins[1]].BoolValue = false;
                        TP.BooleanInput[InversionButtonFBJoins[2]].BoolValue = true;
                        break;
                    }
                }
                //TP.StringInput[index + 10].StringValue = ActiveCAM300.InversionStateFB[(int)index];
            }
            if (args.PropertyName.Contains("CurrentResolutions["))
            {
                //ErrorLog.Notice("Resolution {0} changed to {1}", args.PropertyName.Substring("CurrentResolutions[".Length, 1), args.PropertyValue.ToString());
                uint index = Convert.ToUInt32(args.PropertyName.Substring("CurrentResolutions[".Length, 1));
                TP.StringInput[index + 10].StringValue = ActiveCAM300.CurrentResolutions[(int)index];
            }
        }
        void OnCAM300Debug(object sender, DebugEventArgs args)
        {
            //ErrorLog.Notice(args.DebugString);
        }
    }
}